/*
 * \file: grl_bmpdec_fgs.c
 *
 * Implementation of fgs formar for SVG Bitmap Decoder
 *
 * \component: SVG Bitmap Decoder (SVGBMPDEC)
 *
 * \author: E. Friedrich (efriedrich@de.adit-jv.com)
 *
 * \copyright: (c) 2011 ADIT Corporation
 */

#include "adit_typedef.h"
#include "svg_bitmap_decoder.h"
#include "grl_bitmap_decoder_util.h"
#include "grl_bmpdec_fgs.h"
#include <stdio.h>

SVGError   GRL_BMPDEC_draw_image_FGS( GRL_BMPDEC_dec_info *p_decode_info )
{
    SVGError        ret             = SVG_NO_ERROR;
    SVGImage        *p_image;                   /*  Pointer to Image struct                 */
    SVGUint8        *dst_color_adrs_org;        /*  Draw address origin                     */
    SVGUint8        *dst_color_adrs;            /*  Draw address                            */
    SVGUint32        *p_dst_adrs_hw;             /*  Pointer to draw address       */
    SVGUint8        *p_dst_mem      = NULL;     /* Pointer to destination memory        */
    SVGInt8         *p_read_data;               /*  Pointer to one line of source image     */
    SVGUint32       read_line_idx   = 0;        /*  Read index for one line                 */
    SVGUint16       cnt_ln          = 0;        /*  Line counter variable                   */
    SVGUint16       cnt_cl          = 0;        /*  Column counter variable                 */
    SVGUint16       draw_data       = 0;        /*  Draw data of one pixel (incl. CK)       */
    SVGBmpDecRect   src_clip;

    SVGUint32       dst_mwidth      = 0;                    /* Destination memory width             */
    SVGInt32        dst_xpos        = 0;
    SVGInt32        dst_ypos        = 0;
    SVGUint32       dst_area_width  = 0;                    /* destination x size of target surface */
    SVGUint32       dst_area_height = 0;                    /* destination y size of target surface */
    SVGUint16       src_width       = 0;                    /* Width of source image                */
    SVGUint16       src_height      = 0;                    /* Height of source image               */
    SVGInt16        clip_x          = 0;
    SVGInt16        clip_y          = 0;
    SVGInt16        clip_width      = 0;
    SVGInt16        clip_height     = 0;
    SVGBoolean      is_clipped      = SVG_FALSE;            /* Image has to be clipped              */
    /*
     * Setup bitmap decoding
     */
    if ( NULL != p_decode_info )
    {
        p_image = &(p_decode_info->image);
        /* Handle destination-specific settings */
        switch ( p_image->decode_mode )
        {
            case SVG_BMPDEC_MODE_MEMORY:
            {
                p_dst_mem       = p_image->dest.dest_mem.dest_mem;
                dst_area_width  = p_image->width;    /* Assume width to be the width of the image */
                dst_mwidth      = dst_area_width * GRL_BMPDEC_DEPTH_ARGB8888;
                dst_area_height = p_image->height;   /* Assume height to be the height of the image */
                dst_xpos        = 0;                 /* Position is always at beginning of mem block because */
                dst_ypos        = 0;                 /* otherwise the size might not be sufficient */
                clip_x          = p_image->clip.x;
                clip_y          = p_image->clip.y;
                clip_width      = p_image->clip.width;
                clip_height     = p_image->clip.height;
                break;
            }
            default:
            {
                /* Invalid destination mode */
                ret = SVG_INVALID_OPERATION;
                SVG_BMP_E("Invalid destination mode (not SVG or mem address), function aborted.");
                SVG_BMP_E("SVG_INVALID_OPERATION IN GRL_BMPDEC_DRAW_IMAGE_FGS");
            }
        }

        if ( NULL != p_image->datap )
        {
            /* Check magic number of image file */
            if ( (p_image->datap[GRL_FGS_IDX_MAGIC_NO_0] != 0x01) ||
                 (p_image->datap[GRL_FGS_IDX_MAGIC_NO_1] != 0x00) ||
                 (p_image->datap[GRL_FGS_IDX_MAGIC_NO_2] != 0x00) ||
                 (p_image->datap[GRL_FGS_IDX_MAGIC_NO_3] != 0x00) )
            {
                ret = SVG_INVALID_OPERATION;
            }
            else
            {
                if ( (clip_x == 0) && (clip_y == 0) && (clip_width == 0) && (clip_height == 0) )
                {
                    clip_width   = p_image->width;
                    clip_height  = p_image->height;
                    is_clipped = SVG_FALSE;
                }
                if( (p_image->width < clip_width) || (p_image->height < clip_height))
                {
                    clip_width  = p_image->width;
                    clip_height = p_image->height;
                }
                if( (clip_width == 0) || ( clip_height == 0))
                {
                    clip_width  = p_image->width - clip_x;
                    clip_height = p_image->height - clip_y;
                    if ( (clip_height < 0) || (clip_width < 0))
                    {
                        clip_width  = 0;
                        clip_height = 0;
                    }
                }
                if( (p_image->width < clip_x) || (p_image->height < clip_y))
                {
                    clip_x  = 0;
                    clip_y  = 0;
                }
                src_width       = *(U16*)(void*)(&p_image->datap[GRL_FGS_IDX_WIDTH_LO]);
                src_height      = *(U16*)(void*)(&p_image->datap[GRL_FGS_IDX_HEIGHT_LO]);
                src_clip.x      = clip_x;
                src_clip.y      = clip_y;
                src_clip.width  = clip_width;
                src_clip.height = clip_height;
                /* Clipping setting */
                ret = GRL_BMPDEC_clipping_setup( dst_area_width, dst_area_height,
                                                 &dst_xpos, &dst_ypos,
                                                 &src_clip,
                                                 src_width, src_height,
                                                 &is_clipped);
                if ( (SVG_NO_ERROR == ret) && (p_dst_mem != NULL))
                {

                    /*
                     * Set index to first line which is not clipped
                     * considering that FGS lines start at the bottom.
                     */
                    read_line_idx = FGS_HDR_SIZE + ((( src_height - 1 ) - src_clip.y ) * ( src_width * sizeof(U16) ));

                    dst_color_adrs_org = (SVGUint8 *) (p_dst_mem + dst_xpos*GRL_BMPDEC_DEPTH_ARGB8888+ dst_mwidth*dst_ypos);
                    for ( cnt_ln = 0; cnt_ln < src_clip.height; cnt_ln++ )
                    {
                    	dst_color_adrs = dst_color_adrs_org;

                        p_read_data = (S8*)&p_decode_info->image.datap[read_line_idx];

                        /* Copy a line */
                        for ( cnt_cl = src_clip.x;
                              cnt_cl < (src_clip.x + src_clip.width);
                              cnt_cl++ )
                        {
                            /* Read pixel color (little endian ) */
                            draw_data = *(U16*)(void*)(&p_read_data[cnt_cl * sizeof(U16)]);

                            /* Draw pixel only if it is not colorkeyed */
                            if ( (draw_data != CHANGE_ARGB8888ToRGB565(p_image->color_key)) ||
                                 (SVG_FALSE == p_decode_info->image.attribute.fgs.use_colorkey))
                            {
                            	p_dst_adrs_hw = (SVGUint32*)(void*)dst_color_adrs; /* TODO: This disables type-checking and may cause alignment issues, improve! */
/* tbd dlangner: not performance optimized -> branch within double loop. e.g. do it by function pointer */
                                if(p_decode_info->revert == SVG_TRUE)
                                {
                                	*p_dst_adrs_hw = CHANGE_RGB565ToARGB8888(draw_data);
                                }
                                else
                                {
                                	*p_dst_adrs_hw = CHANGE_RGB565ToABGR8888(draw_data);
                                }
                            }

                            dst_color_adrs = &dst_color_adrs[GRL_BMPDEC_DEPTH_ARGB8888];
                        }

                        dst_color_adrs_org = &dst_color_adrs_org[dst_mwidth];
                        read_line_idx -= src_width*sizeof(U16) ;
                    }

                }
            }
        }
        else
        {
            ret = SVG_POINTER_NULL;
        }
    }
    else
    {
        ret = SVG_POINTER_NULL;
    }

    return ret;
}


SVGError   GRL_BMPDEC_get_image_info_FGS( const SVGContextBmpDec *bmp_ctx,
                                   const SVGImage         *p_image,
                                   SVGImageInfo           *image_info )
{
    SVGError       ret     = SVG_NO_ERROR;

    /* Parameter check */
    if ( (bmp_ctx == NULL) || (p_image == NULL) || (image_info == NULL) || (p_image->datap == NULL) )
    {
        ret = SVG_POINTER_NULL;
    }
    else
    {
        /* Check magic number of image file */
        if ( (p_image->datap[GRL_FGS_IDX_MAGIC_NO_0] != 0x01) ||
             (p_image->datap[GRL_FGS_IDX_MAGIC_NO_1] != 0x00) ||
             (p_image->datap[GRL_FGS_IDX_MAGIC_NO_2] != 0x00) ||
             (p_image->datap[GRL_FGS_IDX_MAGIC_NO_3] != 0x00) )
        {
            ret = SVG_INVALID_OPERATION;
        }
        else
        {
            /* Read info from image */
            image_info->type        = SVG_BMPDEC_FGS_ENCODING;
            image_info->width       = *(U16*)(void*)(&p_image->datap[GRL_FGS_IDX_WIDTH_LO]); /* TODO: Make safer against alignment issues */
            image_info->height      = *(U16*)(void*)(&p_image->datap[GRL_FGS_IDX_HEIGHT_LO]);

        }
    }

    return ret;
}
